classdef SimilarityMatrix < handle
    % SIMILARITYMATRIX Class to store the similarity matrix of a graph.
    %
    % simMat = SimilarityMatrix(adjMat, ... , 'param', paramValue, ...)
    % will create a SimilarityMatrix object using adjMat as the
    % AdjacencyMatrix for the underlying network, with similarity method
    % specified by the parameters
    %
    % Parameters for constructor:
    %   'similarityMethod' - determines which similarity algorithm is used
    %       'cosine'
    %       'nodeSimilarity'
    %       'blondelSimilarity'
    %       'weightSimilarity' (default)
    %       'iterWeightSimilarity'
    %       'tanimoto'
    %
    %   'updateMethod' - this only has an effect when the 'bloniarz'
    %   similarity method is used.  This determines how the similarity
    %   matrix is updated when iterating the calculations.  Should be one
    %   of the following:
    %       'average'
    %       'solution' (default)
    %       'weighted'
    %
    %   'direction' - determine the direction of connection to consider.
    %   Probably should only use change this if using a non-iterative
    %   method (cosine or weightSimilarity)
    %       'incoming'
    %       'outgoing'
    %       'both' (default)
    %
    %   'groups' - should be followed by a cell array of strings which
    %       alternates groups names with a cell array of that group's
    %       contents
    
    
    
    properties (Access = protected)
        %The actual similarity matrix and row/column labels
        S
        Slabels
        
        %The original adjacency matrix
        adjMat
        
        %Handle to function which will calculate similarity matrix
        generateMatrix
        
        %Function parameters
        similarityMethod = 'weightSimilarity';
        updateMethod = 'solution';
        direction = 'both';
        
        %Weights for considering incoming vs. outgoing connections
        inWeight = 1;
        outWeight = 1;
        
        %Groups for identifying neurons
        groups = {};
        
    end
    
    methods (Access = protected)
        % Various ways of calculating similarity
        % Implementations of other methods
        cosSimilarity(obj,varargin)
        simRank(obj,varargin)
        tanimoto(obj,varargin)
        
        % Inhouse methods
        nodeSimilarity(obj,varargin)
        weightSimilarity(obj,varargin)
        iterWeightSimilarity(obj,varargin)
    end
    
    methods (Access = public)
        function obj = SimilarityMatrix(adjMatObj, varargin)
            obj.adjMat = adjMatObj;
            % Set default similarity function
            obj.generateMatrix = @weightSimilarity;
            
            % Read parameters
            for i = 1:2:numel(varargin)
                switch varargin{i}
                    case 'similarityMethod'
                        switch(varargin{i+1})
                            case 'cosine'
                                obj.generateMatrix = @cosSimilarity;
                            case 'bloniarz'
                                obj.generateMatrix = @nodeSimilarity;
                                obj.similarityMethod = 'bloniarz';
                                obj.updateMethod = 'weighted';
                            case 'blondel'
                                obj.generateMatrix = @nodeSimilarity;
                                obj.similarityMethod = 'blondel';
                                obj.updateMethod = 'solution';
                            case 'simRank'
                                obj.generateMatrix = @simRank;
                            case 'weightSimilarity'
                                obj.generateMatrix = @weightSimilarity;
                                obj.similarityMethod = 'weightSimilarity';
                            case 'iterWeightSimilarity'
                                obj.generateMatrix = @iterWeightSimilarity;
                                obj.similarityMethod = 'iterWeightSimilarity';
                                obj.updateMethod = 'solution';
                            case 'tanimoto'
                                obj.generateMatrix = @tanimoto;
                                obj.similarityMethod = 'tanimoto';
                                
                        end
                    case 'direction'
                        obj.direction = varargin{i+1};
                        switch varargin{i+1}
                            case 'incoming'
                                obj.inWeight = 1;
                                obj.outWeight = 0;
                            case 'outgoing'
                                obj.outWeight = 1;
                                obj.inWeight = 0;
                        end
                        
                    case 'groups'
                        obj.groups = varargin{i+1};
                    case 'updateMethod'
                        obj.updateMethod = varargin{i+1};
                        
                end
            end
            obj.generateMatrix(obj,'quiet',false);
        end
        
        function [S Slabels] = getMatrix(obj)
            S = obj.S; Slabels = obj.Slabels;
        end
        
        disp(obj)
        B = subsref(A,S)
        makeDendrogram(obj,labels)
        showSimilarNeurons(obj,neuron,N)
        h = showHistogram(obj,N)
        showMostSimilarPairs(obj,N)
        H = makeHeatmap(obj,labels)
        
    end
    
end